/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.ec.jrc.qcs.engine;

import eu.europa.ec.jrc.qcs.dao.RuleOutputDAO;
import eu.europa.ec.jrc.qcs.dao.RuleOutputDetailDAO;
import eu.europa.ec.jrc.qcs.dao.datasource.DAO;
import eu.europa.ec.jrc.qcs.dao.datasource.DAOFactory;
import eu.europa.ec.jrc.qcs.dao.datasource.FileConnection;
import eu.europa.ec.jrc.qcs.dao.datasource.FlatFileDAOFactory;
import eu.europa.ec.jrc.qcs.dao.file.handler.FileHandlerCSV;
import eu.europa.ec.jrc.qcs.dao.model.MessageType;
import eu.europa.ec.jrc.qcs.dao.model.SerializableModel;
import eu.europa.ec.jrc.qcs.dao.model.output.RuleOutput;
import eu.europa.ec.jrc.qcs.dao.model.output.RuleOutputDetail;
import eu.europa.ec.jrc.qcs.engine.rule.AbstractRule;
import eu.europa.ec.jrc.qcs.test.SystemTester;
import java.io.File;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RawFilesWriter {
    private int validationRunID;
    private int numberOfErrors;
    private int numberOfWarnings;
    private static int latencyIDSpan;
    private int maxNumberOfDetails;
    protected boolean writeOnFile = true;
    protected boolean writeOnDAO = false;
    protected boolean benchmark = false;
    protected FileHandlerCSV<RuleOutput> ruleOutputHandler;
    protected FileHandlerCSV<RuleOutputDetail> ruleOutputDetailHandler;
    protected int lastRuleOutputID;
    protected int lastRuleOutputDetailID;
    protected RuleOutputDAO ruleOutputDAO;
    protected RuleOutputDetailDAO ruleOutputDetailDAO;
    protected static Logger logger;

    public RawFilesWriter() {
        latencyIDSpan = 2;
    }

    protected void init() {
        boolean firstInit;
        if (logger.isTraceEnabled()) {
            logger.trace("init() - Initalising raw files writer");
        }
        DAOFactory daoFactory = DAOFactory.getDAOFactory();
        this.ruleOutputDAO = (RuleOutputDAO)((Object)daoFactory.getDAO(DAO.RULE_OUTPUT));
        this.ruleOutputDetailDAO = (RuleOutputDetailDAO)((Object)daoFactory.getDAO(DAO.RULE_OUTPUT_DETAIL));
        this.ruleOutputDAO.setLastValidationRun(true);
        if (this.writeOnFile) {
            this.ruleOutputHandler = this.initFileHandler(DAO.RULE_OUTPUT, RuleOutput.class);
            this.ruleOutputDetailHandler = this.initFileHandler(DAO.RULE_OUTPUT_DETAIL, RuleOutputDetail.class);
            if (logger.isInfoEnabled()) {
                logger.info("init() - Saving raw files directly on file system (temp folder)");
            }
        }
        if (this.writeOnDAO) {
            if (this.validationRunID > 0) {
                if (logger.isDebugEnabled()) {
                    logger.debug("init() - Initialising DAO components");
                }
                this.ruleOutputDAO.init(this.validationRunID);
                this.ruleOutputDetailDAO.init(this.validationRunID);
                this.ruleOutputDAO.openConnection();
                this.ruleOutputDetailDAO.openConnection();
            } else {
                String message = "init() - DAO objects not initialized: validationRunID is missing";
                logger.warn(message);
            }
            if (logger.isInfoEnabled()) {
                logger.info("init() - Saving raw files through DAO layer");
            }
        }
        if (this.benchmark && !(firstInit = SystemTester.initBenchmark())) {
            SystemTester.benchmark("start.raw.files");
        }
        if (logger.isTraceEnabled()) {
            logger.trace("init() - Initalised raw files writer");
        }
    }

    protected <T extends SerializableModel> FileHandlerCSV<T> initFileHandler(DAO dao, Class<T> type) {
        FileConnection fileConnection = FlatFileDAOFactory.getConnection(dao);
        fileConnection.setWriteAccess(true);
        FileHandlerCSV<SerializableModel> handler = null;
        if (type == RuleOutput.class) {
            handler = new FileHandlerCSV<RuleOutput>(fileConnection, RuleOutput.class, true);
        } else if (type == RuleOutputDetail.class) {
            handler = new FileHandlerCSV<RuleOutputDetail>(fileConnection, RuleOutputDetail.class, true);
        }
        handler.setHeader(true);
        handler.setAutoCommit(false);
        handler.setNewFile(true);
        handler.openWriter(this.validationRunID);
        this.lastRuleOutputID = this.ruleOutputDAO.getMaxID();
        this.lastRuleOutputDetailID = this.ruleOutputDetailDAO.getMaxID();
        if (logger.isDebugEnabled()) {
            logger.debug("initFileHandler() - Created new handler: " + String.valueOf(handler) + "(autocomit=" + handler.isAutoCommit() + ")");
        }
        return handler;
    }

    public void reset() {
        this.numberOfErrors = 0;
        this.numberOfWarnings = 0;
        this.maxNumberOfDetails = 0;
    }

    protected void saveRuleOutput(List<RuleOutput> output) {
        MessageType messageType;
        if (this.writeOnDAO) {
            if (logger.isTraceEnabled()) {
                logger.trace("saveRuleOutput()- Saving output records through DAO");
            }
            this.ruleOutputDAO.save(output);
        }
        if (this.writeOnFile) {
            if (logger.isTraceEnabled()) {
                logger.trace("saveRuleOutput()- Saving output records through File Handler");
            }
            for (RuleOutput ruleOutput : output) {
                ruleOutput.setId(++this.lastRuleOutputID);
            }
            this.ruleOutputHandler.writeFile(this.validationRunID, output);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("saveRuleOutput() - " + output.size());
        }
        if (output.size() > 0) {
            RuleOutput firstRuleOutput = output.get(0);
            messageType = firstRuleOutput.getMessageType();
        } else {
            messageType = MessageType.CRITICAL;
            logger.error("saveRuleOutput() - One protocol's rule produced a NULL output");
        }
        switch (messageType) {
            case WARNING: {
                this.numberOfWarnings += output.size();
                break;
            }
            default: {
                this.numberOfErrors += output.size();
            }
        }
    }

    public void saveRuleOutputDetail(AbstractRule rule, List<RuleOutput> output) {
        this.saveRuleOutputDetail(rule, 0, output);
    }

    public void saveRuleOutputDetail(AbstractRule rule, int lineNumber, List<RuleOutput> output) {
        List<RuleOutputDetail> outputDetails;
        if (this.benchmark) {
            this.benchmark("__PRE_SAVE", rule, lineNumber);
        }
        this.saveRuleOutput(output);
        if (this.benchmark) {
            this.benchmark("__POST_SAVE", rule, lineNumber);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("saveRuleOutputDetail() - Checking if to write on DAO");
        }
        if (this.writeOnDAO) {
            for (RuleOutput ruleOutput : output) {
                outputDetails = ruleOutput.getAllRuleOutputDetail();
                if (outputDetails == null) continue;
                this.ruleOutputDetailDAO.save(outputDetails);
            }
        }
        if (logger.isDebugEnabled()) {
            logger.debug("saveRuleOutputDetail() - Checking if to write on file");
        }
        if (this.writeOnFile) {
            for (RuleOutput ruleOutput : output) {
                outputDetails = ruleOutput.getAllRuleOutputDetail();
                if (outputDetails != null) {
                    for (RuleOutputDetail ruleOutputDetail : outputDetails) {
                        ruleOutputDetail.setId(++this.lastRuleOutputDetailID);
                    }
                    this.ruleOutputDetailHandler.writeFile(this.validationRunID, outputDetails);
                    if (logger.isTraceEnabled()) {
                        logger.trace("saveRuleOutputDetail(): RuleOutputID " + ruleOutput.getId() + " -> number of details: " + outputDetails.size());
                    }
                    if (outputDetails.size() <= this.maxNumberOfDetails) continue;
                    this.maxNumberOfDetails = outputDetails.size();
                    continue;
                }
                if (!logger.isDebugEnabled()) continue;
                logger.debug("saveRuleOutputDetail(): no details for RuleOutput: " + String.valueOf(ruleOutput));
            }
        }
        if (this.benchmark) {
            this.benchmark("__DETAIL", rule, lineNumber);
        }
    }

    public void commit() {
        if (this.writeOnDAO) {
            this.ruleOutputDAO.closeConnection();
            this.ruleOutputDetailDAO.closeConnection();
            if (logger.isDebugEnabled()) {
                logger.debug("commit() - Closed all connections to DAO layer");
            }
        }
        if (this.writeOnFile) {
            this.ruleOutputHandler.closeWriter();
            this.ruleOutputDetailHandler.closeWriter();
            if (logger.isDebugEnabled()) {
                logger.debug("commit() - Closed all connections");
            }
        }
    }

    public void checkTempFolder() {
        File tempFolder = this.getTemporaryFolder();
        if (tempFolder.exists() && tempFolder.isDirectory()) {
            if (logger.isDebugEnabled()) {
                logger.debug("checkTempFolder()- Found temp folder: " + tempFolder.getAbsolutePath());
            }
        } else {
            tempFolder.mkdir();
            if (logger.isDebugEnabled()) {
                logger.debug("checkTempFolder()- Created temp folder: " + tempFolder.getAbsolutePath());
            }
        }
    }

    public void cleanTempFolder(int maxValidationRunID) {
        int maximumTestID = 3;
        File tempFolder = this.getTemporaryFolder();
        if (tempFolder.exists() && tempFolder.isDirectory()) {
            File[] filesList;
            if (logger.isDebugEnabled()) {
                logger.debug("cleanTempFolder()- Cleaning very old raw files from temp folder using last validationRunID = " + maxValidationRunID);
            }
            for (File file : filesList = tempFolder.listFiles()) {
                int resourceID;
                if (!file.isFile()) continue;
                String fileName = file.getName();
                String fileSuffix = fileName.replaceAll("\\D+", "");
                if (logger.isDebugEnabled()) {
                    logger.debug("cleanTempFolder()- Checking latency for file: " + fileName + " -> " + fileSuffix);
                }
                if (fileSuffix == null || fileSuffix.length() <= 0) continue;
                try {
                    resourceID = Integer.parseInt(fileSuffix);
                }
                catch (NumberFormatException e) {
                    resourceID = -1;
                }
                if (resourceID >= 0 && resourceID <= maximumTestID) continue;
                int spanID = 1 + maxValidationRunID - resourceID;
                if (spanID >= latencyIDSpan) {
                    file.delete();
                }
                if (!logger.isDebugEnabled()) continue;
                logger.debug("cleanTempFolder() - Checked old raw file: " + fileName + " (spanID=" + spanID + ")");
            }
        } else {
            logger.warn("cleanTempFolder()- Temp folder not found: nothing to clean");
        }
    }

    protected File getTemporaryFolder() {
        FileConnection validationRunConnection = FlatFileDAOFactory.getConnection(DAO.VALIDATION_RUN);
        String tempPath = validationRunConnection.getRelativePath();
        String userDir = System.getProperty("user.dir");
        String actualPath = userDir + "/" + tempPath;
        File tempFolder = new File(actualPath);
        if (logger.isTraceEnabled()) {
            logger.trace("getTemporaryFolder() - Temporary folder: " + actualPath);
        }
        return tempFolder;
    }

    public void setValidationRunID(int validationRunID) {
        this.validationRunID = validationRunID;
        this.init();
    }

    public int getNumberOfMessages() {
        int total = this.getNumberOfErrors() + this.getNumberOfWarnings();
        return total;
    }

    public int getValidationRunID() {
        return this.validationRunID;
    }

    public boolean isWriteOnDAO() {
        return this.writeOnDAO;
    }

    public void setWriteOnDAO(boolean writeOnDAO) {
        this.writeOnDAO = writeOnDAO;
    }

    public boolean isWriteOnFile() {
        return this.writeOnFile;
    }

    public void setWriteOnFile(boolean writeOnFile) {
        this.writeOnFile = writeOnFile;
    }

    public int getNumberOfErrors() {
        return this.numberOfErrors;
    }

    public int getNumberOfWarnings() {
        return this.numberOfWarnings;
    }

    public boolean isBenchmark() {
        return this.benchmark;
    }

    public void setBenchmark(boolean benchmark) {
        this.benchmark = benchmark;
    }

    public int getLastRuleOutputID() {
        return this.lastRuleOutputID;
    }

    public int getLastRuleOutputDetailID() {
        return this.lastRuleOutputDetailID;
    }

    public int getMaxNumberOfDetails() {
        return this.maxNumberOfDetails;
    }

    public static int getLatencyIDSpan() {
        return latencyIDSpan;
    }

    public static void setLatencyIDSpan(int latencyIDSpan) {
        RawFilesWriter.latencyIDSpan = latencyIDSpan;
    }

    protected void benchmark(String prefix, AbstractRule rule, int lineNumber) {
        boolean logTime = true;
        if (logTime) {
            StringBuilder ruleIdentifier = new StringBuilder();
            StringBuilder recordIdentifier = new StringBuilder();
            ruleIdentifier.append(prefix);
            ruleIdentifier.append("_");
            ruleIdentifier.append(rule.getId());
            ruleIdentifier.append(": ");
            ruleIdentifier.append(rule.getName());
            recordIdentifier.append(" (line=");
            recordIdentifier.append(lineNumber);
            recordIdentifier.append(")");
            SystemTester.benchmark(ruleIdentifier.toString(), recordIdentifier.toString());
        }
    }

    static {
        logger = LoggerFactory.getLogger(RawFilesWriter.class);
    }
}

